/* $Id: fourtach.c,v 1.2 1999/03/12 23:45:10 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test program for four tach signals.  This finds a problem that used
   to exist when there are no active inputs in a module, but there are
   active tach channels.  You can check for the problem by doing:
   "./fourtach -v -n0", or by doing "./fourtach -v -N2 -n8". */

/* This test program can be used with one input module, or with two
   input modules.  All modules should have tach boards.  The tach
   signals should be:

   Tach0, Tach3:	100 Hz -> 500 Hz, 30 sec, square wave, 1 volt peak
   Tach1, Tach2:	 50 Hz -> 250 Hz, 30 sec, square wave, 1 volt peak

   The signal generator should be in continuous sweep mode.

   This setup can be done with a single 3326.  Without one of those, I
   think you're stuck.

   The setup for this test program is identical to the setup for the
   rpmarm test program, and both are called from the rpmarm-suite QA
   test suite. */

#include <math.h>		/* For fabs */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	BLOCKSIZE	1024
#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	NTCHAN_MAX	4

/* Use a *big* margin, to avoid false errors */
#define	RPM_MARGIN	150
#define	RPM_MAX		30000

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: fourtach.c,v 1.2 1999/03/12 23:45:10 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, int *ntchan, SHORTSIZ16 *chan_list, int *mod_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, j, nc, nt, chan;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    nt = 0;
    for (i = 0; i < nmod; i++)
    {
	nc += hwconfig[i].input_chans;
	nt += hwconfig[i].tach_chans;
    }
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    if (nt > NTCHAN_MAX)
	nt = NTCHAN_MAX;
    if (nt > *ntchan && *ntchan != -1)
	nt = *ntchan;
    *nchan = nc;
    *ntchan = nt;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);
    for (i = 0; i < nt; i++)
	chan_list[i + nc] = E1432_TACH_CHAN(i + 1);

    chan = 0;
    for (i = 0; i < nmod; i++)
	for (j = 0; j < hwconfig[i].input_chans; j++, chan++)
	{
	    if (chan >= NCHAN_MAX)
		break;
	    mod_list[chan] = i;
	}

    *group = e1432_create_channel_group(*hw, nc + nt, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int freq, int magsq)
{
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, E1432_BLOCK_MODE));

    /* Try to handle either TTL levels or +-1V input */
    CHECK(e1432_set_trigger_level(hw, group,
				  E1432_TRIGGER_LEVEL_LOWER, 0.4));
    CHECK(e1432_set_trigger_level(hw, group,
				  E1432_TRIGGER_LEVEL_UPPER, 0.8));

    if (freq)
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
	if (magsq)
	{
	    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_RMS));
	    CHECK(e1432_set_avg_number(hw, group, 1));
	}
    }

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
#if 0
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
#endif
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

#define	ZERO_NOT_ALLOWED	0x0
#define	ZERO_ALLOWED		0x1
#define	ZERO_REQUIRED		0x2

static int
check_rpm(double ref, double rpm_to_check, int zero_flags)
{
    double diff = fabs(ref - rpm_to_check);

    if (zero_flags & ZERO_REQUIRED)
    {
	if (rpm_to_check != 0)
	{
	    (void) fprintf(stderr, "%s: rpm %g: expected zero\n",
			   progname, rpm_to_check);
	    return -1;
	}
    }
    else if (zero_flags & ZERO_ALLOWED)
    {
	if (diff > RPM_MARGIN && rpm_to_check != 0)
	{
	    (void) fprintf(stderr, "%s: rpm %g: expected %g or zero\n",
			   progname, rpm_to_check, ref);
	    return -1;
	}
    }
    else
    {
	if (diff > RPM_MARGIN)
	{
	    (void) fprintf(stderr, "%s: rpm %g: expected %g\n",
			   progname, rpm_to_check, ref);
	    return -1;
	}
    }

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, int ntchan, SHORTSIZ16 *chan_list,
    int *mod_list, long nscan, int freq, int magsq, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    FLOATSIZ32 crpm[NTCHAN_MAX], drpm[NTCHAN_MAX], erpm[NTCHAN_MAX];
    LONGSIZ32 count, expect;
    struct e1432_trailer trailer;
    double  prev_crpm0;
    int     i, scan, type, type_max, chan;

    CHECK(e1432_init_measure(hw, group));

    (void) memset(crpm, 0, sizeof crpm);
    (void) memset(drpm, 0, sizeof drpm);
    (void) memset(erpm, 0, sizeof erpm);
    prev_crpm0 = 0;

    for (scan = 0; scan < nscan; scan++)
    {
	/* Wait for block available */
	CHECK(wait_block_avail(hw, group, scan, verbose,
			       BLOCKSIZE, 20000));

	for (i = 0; i < ntchan; i++)
	{
	    CHECK(e1432_get_current_rpm(hw, chan_list[nchan + i], &crpm[i]));
	    CHECK(e1432_get_data_rpm(hw, chan_list[nchan + i], &erpm[i]));
	}

	type_max = freq ? 1 : 0;

	/* Read the data */
	for (type = 0; type <= type_max; type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (verbose > 1)
		    (void) printf("Reading chan %d type %d\n",
				  chan, type);

		expect = BLOCKSIZE;
		if (type != 0 && magsq)
		    expect = BLOCKSIZE / 2;
		CHECK(e1432_read_float64_data(hw, chan_list[chan],
					      type == 0 ?
					      E1432_TIME_DATA :
					      E1432_FREQ_DATA,
					      buffer, expect,
					      &trailer,
					      &count));
		if (count != expect)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "actual count was %ld\n",
				   progname, count);
		    return -1;
		}

		CHECK(check_trailer(&trailer, 51200, 20000,
				    chan_list[chan], type));

#if	NTCHAN_MAX >= 2
		if (mod_list[chan] == 0)
		{
		    drpm[0] = trailer.rpm1;
		    drpm[1] = trailer.rpm2;
		}
#endif
#if	NTCHAN_MAX >= 4
		if (mod_list[chan] == 1)
		{
		    drpm[2] = trailer.rpm1;
		    drpm[3] = trailer.rpm2;
		}
#endif
	    }
	if (verbose > 0 && ntchan >= 3)
	    (void) printf("C %5.0f %5.0f %5.0f %5.0f  "
			  "D %5.0f %5.0f %5.0f %5.0f  "
			  "E %5.0f %5.0f %5.0f %5.0f\n",
			  crpm[0], crpm[1], crpm[2], crpm[3],
			  drpm[0], drpm[1], drpm[2], drpm[3],
			  erpm[0], erpm[1], erpm[2], erpm[3]);
	else if (verbose > 0 && ntchan >= 1)
	    (void) printf("C %5.0f %5.0f  D %5.0f %5.0f  E %5.0f %5.0f\n",
			  crpm[0], crpm[1], drpm[0], drpm[1],
			  erpm[0], erpm[1]);

	for (i = 0; i < ntchan; i++)
	{
	    /* The first scan, and occasionally the second scan, can
               have zero data RPM, allow that. */
	    if (prev_crpm0 > RPM_MAX - RPM_MARGIN)
	    {
		/* Previous current RPM was near the max.  The next
		   data RPM may not be a close match to the current
		   RPM, so check only that the two data RPMs are close
		   to each other. */
		/* If mod_list shows us with active input channels
		   in only the first module, don't use drpm from the
		   second module. */
		if (nchan > 0 && (i < 2 || mod_list[nchan - 1] > 0))
		    CHECK(check_rpm(drpm[i], erpm[i],
				    scan < 2 ? ZERO_ALLOWED :
				    ZERO_NOT_ALLOWED));
	    }
	    else
	    {
		/* Normal case */
		/* If mod_list shows us with active input channels
		   in only the first module, don't use drpm from the
		   second module. */
		if (nchan > 0 && (i < 2 || mod_list[nchan - 1] > 0))
		    CHECK(check_rpm(crpm[i], drpm[i],
				    scan < 2 ? ZERO_ALLOWED :
				    ZERO_NOT_ALLOWED));
		/* But use erpm even if drpm is not valid */
		CHECK(check_rpm(crpm[i], erpm[i],
				scan < 2 ? ZERO_ALLOWED :
				ZERO_NOT_ALLOWED));
	    }

	    /* Tach chan 0 should match tach chan 3, and
	       tach chan 1 should match tach chan 2, assuming the
	       signals are set up correctly. */
	    if (i == 2)
		CHECK(check_rpm(crpm[1], crpm[2], ZERO_NOT_ALLOWED));
	    if (i == 3)
		CHECK(check_rpm(crpm[0], crpm[3], ZERO_NOT_ALLOWED));
	}
	for (; i < NTCHAN_MAX; i++)
	{
	    CHECK(check_rpm(crpm[i], crpm[i], ZERO_REQUIRED));
	    CHECK(check_rpm(crpm[i], drpm[i], ZERO_REQUIRED));
	    CHECK(check_rpm(crpm[i], erpm[i], ZERO_REQUIRED));
	}
	prev_crpm0 = crpm[0];
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-fmuvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t[-s nscan] [-t ntchan]\n"
		   "\t-f: Set up for freq data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Send mag-squared data, not complex, for freq data\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-s: Run for <scans> scans, default 100\n"
		   "\t-t: Use <ntchan> tach channels, default -1 meaning all\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX + NTCHAN_MAX];
    int     mod_list[NCHAN_MAX];
    char   *p;
    long    nscan;
    int     c, i, freq, magsq, nmod, verbose;
    int     group, nchan, ntchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    freq = 0;
    laddr[0] = 8;
    magsq = 0;
    nchan = -1;			/* Meaning use all input channels */
    ntchan = -1;		/* Meaning use all tach channels */
    nmod = 1;
    nscan = 100;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "fL:mn:N:s:t:uvV")) != -1)
	switch (c)
	{
	case 'f':
	    freq = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    magsq = 1;
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    nscan = strtol(optarg, &p, 0);
	    if (optarg == p || nscan < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of scans: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 't':
	    ntchan = strtol(optarg, &p, 0);
	    if (optarg == p || ntchan < -1 || ntchan > NTCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of tach channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan,
	     &ntchan, chan_list, mod_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, freq, magsq) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, ntchan, chan_list, mod_list,
	    nscan, freq, magsq, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
